/*
 * XORCE - XOR Algebra Compiler
 *
 * Pass 7: Main compiler driver
 * Integrates lexer, parser, holonomy engine, and code generator.
 *
 * Zero external dependencies (libc only).
 */

#define _POSIX_C_SOURCE 200809L

#include "types.h"
#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <ctype.h>

/* ============================================================================
 * VERSION & USAGE
 * ============================================================================ */

#define XORCE_VERSION_MAJOR 1
#define XORCE_VERSION_MINOR 0

static void version(void)
{
    printf("XORCE v%d.%d - XOR Algebra Compiler\n",
           XORCE_VERSION_MAJOR, XORCE_VERSION_MINOR);
}

static void usage(void)
{
    version();
    printf("\nUsage:\n");
    printf("  xorce <source.xor>           Compile source file\n");
    printf("  xorce -r | --repl            Start interactive REPL\n");
    printf("  xorce -h | --help            Show this help message\n");
    printf("  xorce -v | --version         Show version\n");
    printf("  xorce -i <file.xorc>         Inspect chip file\n");
    printf("  xorce --verify <file.xorc>   Verify chip seal\n");
}

/* ============================================================================
 * FILE READING
 * ============================================================================ */

/*
 * Read entire file into a malloc'd string.
 * Returns NULL on error.
 */
static char* read_file(const char *path)
{
    FILE *f = fopen(path, "rb");
    if (!f) {
        return NULL;
    }

    /* Get file size */
    fseek(f, 0, SEEK_END);
    long size = ftell(f);
    fseek(f, 0, SEEK_SET);

    if (size < 0) {
        fclose(f);
        return NULL;
    }

    /* Allocate buffer */
    char *buffer = (char*)malloc((size_t)size + 1);
    if (!buffer) {
        fclose(f);
        return NULL;
    }

    /* Read file */
    size_t read_count = fread(buffer, 1, (size_t)size, f);
    fclose(f);

    buffer[read_count] = '\0';
    return buffer;
}

/* ============================================================================
 * KERNEL REGISTRY INFO STORAGE
 *
 * We need to track the family and params for each kernel so that export
 * can generate proper chip files.
 * ============================================================================ */

#define MAX_KERNEL_INFO 64

typedef struct {
    char *name;
    xorc_family_t family;
    kernel_spec_t spec;
} kernel_info_t;

static kernel_info_t kernel_info[MAX_KERNEL_INFO];
static int kernel_info_count = 0;

static void store_kernel_info(const char *name, kernel_spec_t *spec)
{
    if (kernel_info_count >= MAX_KERNEL_INFO) return;

    kernel_info[kernel_info_count].name = strdup(name);
    kernel_info[kernel_info_count].family = spec->family;
    /* Copy the spec for later use */
    memcpy(&kernel_info[kernel_info_count].spec, spec, sizeof(kernel_spec_t));
    kernel_info[kernel_info_count].spec.name = strdup(spec->name ? spec->name : name);
    kernel_info_count++;
}

static kernel_info_t* find_kernel_info(const char *name)
{
    for (int i = 0; i < kernel_info_count; i++) {
        if (strcmp(kernel_info[i].name, name) == 0) {
            return &kernel_info[i];
        }
    }
    return NULL;
}

static void free_kernel_info(void)
{
    for (int i = 0; i < kernel_info_count; i++) {
        free(kernel_info[i].name);
        free(kernel_info[i].spec.name);
    }
    kernel_info_count = 0;
}

/* ============================================================================
 * FAMILY NAME UTILITIES
 * ============================================================================ */

static const char* family_name(xorc_family_t family)
{
    switch (family) {
        case XORC_FLAT:     return "flat";
        case XORC_PAULI:    return "pauli";
        case XORC_CLIFFORD: return "clifford";
        case XORC_CAYLEY:   return "cayley";
        case XORC_CUSTOM:   return "custom";
        default:            return "unknown";
    }
}

static const char* property_name(property_t prop)
{
    switch (prop) {
        case PROP_ASSOCIATIVE: return "associative";
        case PROP_COMMUTATIVE: return "commutative";
        case PROP_CENTER:      return "center";
        case PROP_RADICAL:     return "radical";
        default:               return "unknown";
    }
}

/* ============================================================================
 * PROCESSING FUNCTIONS
 * ============================================================================ */

/*
 * Process a kernel declaration.
 * Builds the kernel, computes holonomy, and registers it.
 */
static void process_kernel(ast_t *node)
{
    kernel_spec_t *spec = &node->data.kernel.spec;
    const char *name = node->data.kernel.name;

    /* Build the kernel */
    xorc_omega_t *omega = build_kernel(spec);
    if (!omega) {
        fprintf(stderr, "  error: failed to build kernel '%s'\n", name);
        return;
    }

    /* Compute holonomy profile */
    xorc_holo_t *holo = holo_compute(omega);
    if (!holo) {
        fprintf(stderr, "  error: failed to compute holonomy for '%s'\n", name);
        omega_free(omega);
        return;
    }

    /* Register the kernel */
    kernel_register(name, omega, holo);

    /* Store info for export */
    store_kernel_info(name, spec);

    /* Print kernel info */
    printf("  kernel '%s': %s dim=%d (%u elements)\n",
           name, family_name(spec->family), spec->dim, omega->size);
    printf("    -> holonomy: %u assoc failures, %u comm failures\n",
           holo->assoc_failures, holo->comm_failures);
}

/*
 * Process a verify statement.
 * Returns true on pass, false on fail.
 */
static bool process_verify(ast_t *node)
{
    const char *name = node->data.verify.name;
    property_t prop = node->data.verify.prop;
    uint32_t value = node->data.verify.value;
    int line = node->line;

    xorc_omega_t *omega = kernel_lookup(name);
    if (!omega) {
        fprintf(stderr, "  error: line %d: kernel '%s' not found\n", line, name);
        return false;
    }

    xorc_holo_t *holo = kernel_holo(name);
    if (!holo) {
        fprintf(stderr, "  error: line %d: holonomy for '%s' not found\n", line, name);
        return false;
    }

    bool passed = false;
    uint32_t actual = 0;

    switch (prop) {
        case PROP_ASSOCIATIVE:
            passed = holo_is_associative(holo);
            actual = holo->assoc_failures;
            break;
        case PROP_COMMUTATIVE:
            passed = holo_is_commutative(holo);
            actual = holo->comm_failures;
            break;
        case PROP_CENTER:
            actual = holo_center_dim(omega);
            passed = (actual == value);
            break;
        case PROP_RADICAL:
            actual = holo_radical_dim(omega);
            passed = (actual == value);
            break;
    }

    if (passed) {
        if (prop == PROP_CENTER || prop == PROP_RADICAL) {
            printf("  verify '%s': %s=%u PASS\n", name, property_name(prop), value);
        } else {
            printf("  verify '%s': %s PASS\n", name, property_name(prop));
        }
    } else {
        if (prop == PROP_CENTER || prop == PROP_RADICAL) {
            fprintf(stderr, "  error: line %d: verify '%s': %s FAIL (expected %u, got %u)\n",
                    line, name, property_name(prop), value, actual);
        } else {
            fprintf(stderr, "  error: line %d: verify '%s': %s FAIL (%u failures)\n",
                    line, name, property_name(prop), actual);
        }
    }

    return passed;
}

/*
 * Process an export statement.
 * Exports the kernel to a .xorc chip file.
 */
static void process_export(ast_t *node)
{
    const char *name = node->data.export.name;
    const char *path = node->data.export.path;

    xorc_omega_t *omega = kernel_lookup(name);
    if (!omega) {
        fprintf(stderr, "  error: kernel '%s' not found\n", name);
        return;
    }

    xorc_holo_t *holo = kernel_holo(name);
    if (!holo) {
        fprintf(stderr, "  error: holonomy for '%s' not found\n", name);
        return;
    }

    kernel_info_t *info = find_kernel_info(name);
    xorc_family_t family = info ? info->family : XORC_FLAT;
    void *params = NULL;

    /* Prepare family-specific params */
    uint8_t param_buf[2] = {0, 0};
    if (info) {
        switch (family) {
            case XORC_PAULI:
                param_buf[0] = info->spec.params.pauli.n;
                params = param_buf;
                break;
            case XORC_CLIFFORD:
                param_buf[0] = info->spec.params.clifford.p;
                param_buf[1] = info->spec.params.clifford.q;
                params = param_buf;
                break;
            case XORC_CAYLEY:
                param_buf[0] = info->spec.params.cayley.level;
                params = param_buf;
                break;
            default:
                break;
        }
    }

    /* Get file size calculation: header + omega table */
    size_t file_size = 64 + (size_t)omega->size * omega->size;

    if (codegen_chip(path, omega, holo, family, params)) {
        printf("  export '%s' -> %s (%zu bytes, sealed)\n", name, path, file_size);
    } else {
        fprintf(stderr, "  error: failed to export '%s' to '%s'\n", name, path);
    }
}

/* ============================================================================
 * REPL MODE
 * ============================================================================ */

/* REPL state */
typedef struct {
    xorc_omega_t *omega;      /* Currently loaded omega */
    char *chip_name;          /* Name of loaded chip */
    xorc_dim_t dim;           /* Dimension */
    xorc_family_t family;     /* Family type */
    uint32_t assoc_fail;      /* Associator failures */
    uint32_t comm_fail;       /* Commutator failures */
} repl_state_t;

static void repl_help(void)
{
    printf("XORCE REPL Commands:\n");
    printf("  load <file.xorc>    Load a chip file\n");
    printf("  mul <u> <v>         Compute delta_u * delta_v\n");
    printf("  assoc <u> <v> <w>   Test associativity at (u,v,w)\n");
    printf("  comm <u> <v>        Test commutativity at (u,v)\n");
    printf("  table               Print omega table (small dims only)\n");
    printf("  info                Show loaded chip info\n");
    printf("  center              List center elements\n");
    printf("  radical             List radical elements\n");
    printf("  help                Show this help\n");
    printf("  quit | exit         Exit REPL\n");
}

static void repl_load(repl_state_t *state, const char *path)
{
    if (!path || strlen(path) == 0) {
        printf("Error: no file specified\n");
        return;
    }

    /* Free existing omega */
    if (state->omega) {
        omega_free(state->omega);
        state->omega = NULL;
    }
    if (state->chip_name) {
        free(state->chip_name);
        state->chip_name = NULL;
    }

    /* Load new chip */
    state->omega = load_chip(path);
    if (!state->omega) {
        printf("Error: failed to load '%s'\n", path);
        return;
    }

    /* Get metadata */
    if (!chip_info(path, &state->dim, &state->family, &state->assoc_fail, &state->comm_fail)) {
        printf("Warning: could not read chip metadata\n");
    }

    /* Store name */
    state->chip_name = strdup(path);

    printf("Loaded: %s, dim=%d, %u elements\n", path, state->omega->dim, state->omega->size);
}

static void repl_mul(repl_state_t *state, xorc_addr_t u, xorc_addr_t v)
{
    if (!state->omega) {
        printf("Error: no chip loaded\n");
        return;
    }

    if (u >= state->omega->size || v >= state->omega->size) {
        printf("Error: index out of range (max %u)\n", state->omega->size - 1);
        return;
    }

    xorc_elem_t result = elem_mul(state->omega, u, v);
    printf("delta_%u * delta_%u = %cdelta_%u\n",
           u, v, result.coeff > 0 ? '+' : '-', result.basis);
}

static void repl_assoc(repl_state_t *state, xorc_addr_t u, xorc_addr_t v, xorc_addr_t w)
{
    if (!state->omega) {
        printf("Error: no chip loaded\n");
        return;
    }

    uint32_t max = state->omega->size - 1;
    if (u > max || v > max || w > max) {
        printf("Error: index out of range (max %u)\n", max);
        return;
    }

    xorc_sign_t assoc = elem_assoc(state->omega, u, v, w);

    /* Compute both bracketing results for display */
    xorc_elem_t ab = elem_mul(state->omega, u, v);
    xorc_elem_t ab_c = elem_mul2(state->omega, ab, (xorc_elem_t){1, w});

    xorc_elem_t bc = elem_mul(state->omega, v, w);
    xorc_elem_t a_bc = elem_mul2(state->omega, (xorc_elem_t){1, u}, bc);

    printf("(delta_%u * delta_%u) * delta_%u = %cdelta_%u\n",
           u, v, w, ab_c.coeff > 0 ? '+' : '-', ab_c.basis);
    printf("delta_%u * (delta_%u * delta_%u) = %cdelta_%u\n",
           u, v, w, a_bc.coeff > 0 ? '+' : '-', a_bc.basis);

    if (assoc == 1) {
        printf("  [associative]\n");
    } else {
        printf("  [NON-associative, differ by sign]\n");
    }
}

static void repl_comm(repl_state_t *state, xorc_addr_t u, xorc_addr_t v)
{
    if (!state->omega) {
        printf("Error: no chip loaded\n");
        return;
    }

    if (u >= state->omega->size || v >= state->omega->size) {
        printf("Error: index out of range (max %u)\n", state->omega->size - 1);
        return;
    }

    xorc_sign_t comm = elem_comm(state->omega, u, v);
    xorc_elem_t uv = elem_mul(state->omega, u, v);
    xorc_elem_t vu = elem_mul(state->omega, v, u);

    printf("delta_%u * delta_%u = %cdelta_%u\n",
           u, v, uv.coeff > 0 ? '+' : '-', uv.basis);
    printf("delta_%u * delta_%u = %cdelta_%u\n",
           v, u, vu.coeff > 0 ? '+' : '-', vu.basis);

    if (comm == 1) {
        printf("  [commute]\n");
    } else {
        printf("  [anticommute]\n");
    }
}

static void repl_table(repl_state_t *state)
{
    if (!state->omega) {
        printf("Error: no chip loaded\n");
        return;
    }

    if (state->omega->dim > 4) {
        printf("Table too large to display (dim=%d > 4)\n", state->omega->dim);
        return;
    }

    uint32_t size = state->omega->size;

    /* Header row */
    printf("    |");
    for (uint32_t v = 0; v < size; v++) {
        printf(" %2u", v);
    }
    printf("\n----+");
    for (uint32_t v = 0; v < size; v++) {
        printf("---");
    }
    printf("\n");

    /* Data rows */
    for (uint32_t u = 0; u < size; u++) {
        printf(" %2u |", u);
        for (uint32_t v = 0; v < size; v++) {
            xorc_sign_t s = omega_get(state->omega, u, v);
            printf("  %c", s > 0 ? '+' : '-');
        }
        printf("\n");
    }
}

static void repl_info(repl_state_t *state)
{
    if (!state->omega) {
        printf("No chip loaded\n");
        return;
    }

    const char *family_names[] = {"FLAT", "PAULI", "CLIFFORD", "CAYLEY", "CUSTOM"};
    const char *fam = (state->family <= XORC_CUSTOM) ? family_names[state->family] : "UNKNOWN";

    printf("Chip:        %s\n", state->chip_name ? state->chip_name : "(unknown)");
    printf("Dimension:   %d\n", state->omega->dim);
    printf("Size:        %u elements\n", state->omega->size);
    printf("Family:      %s\n", fam);
    printf("Assoc fail:  %u\n", state->assoc_fail);
    printf("Comm fail:   %u\n", state->comm_fail);

    /* Compute center and radical dimensions */
    uint32_t center_dim = holo_center_dim(state->omega);
    uint32_t radical_dim = holo_radical_dim(state->omega);
    printf("Center:      %u elements\n", center_dim);
    printf("Radical:     %u elements\n", radical_dim);
}

static void repl_center(repl_state_t *state)
{
    if (!state->omega) {
        printf("Error: no chip loaded\n");
        return;
    }

    xorc_addr_t *center_list = (xorc_addr_t*)malloc(state->omega->size * sizeof(xorc_addr_t));
    if (!center_list) {
        printf("Error: allocation failed\n");
        return;
    }

    uint32_t count = holo_center_list(state->omega, center_list);
    printf("Center (%u elements):", count);
    for (uint32_t i = 0; i < count; i++) {
        printf(" %u", center_list[i]);
    }
    printf("\n");

    free(center_list);
}

static void repl_radical(repl_state_t *state)
{
    if (!state->omega) {
        printf("Error: no chip loaded\n");
        return;
    }

    xorc_addr_t *radical_list = (xorc_addr_t*)malloc(state->omega->size * sizeof(xorc_addr_t));
    if (!radical_list) {
        printf("Error: allocation failed\n");
        return;
    }

    uint32_t count = holo_radical_list(state->omega, radical_list);
    printf("Radical (%u elements):", count);
    for (uint32_t i = 0; i < count; i++) {
        printf(" %u", radical_list[i]);
    }
    printf("\n");

    free(radical_list);
}

static void repl_free_state(repl_state_t *state)
{
    if (state->omega) {
        omega_free(state->omega);
        state->omega = NULL;
    }
    if (state->chip_name) {
        free(state->chip_name);
        state->chip_name = NULL;
    }
}

static int run_repl(void)
{
    version();
    printf("Interactive mode. Type 'help' for commands.\n\n");

    repl_state_t state = {0};
    char line[256];

    while (1) {
        printf("xorce> ");
        fflush(stdout);

        if (!fgets(line, sizeof(line), stdin)) {
            /* EOF */
            printf("\n");
            break;
        }

        /* Remove trailing newline */
        size_t len = strlen(line);
        if (len > 0 && line[len-1] == '\n') {
            line[len-1] = '\0';
            len--;
        }

        /* Skip empty lines */
        if (len == 0) continue;

        /* Parse command */
        char cmd[32] = {0};
        char arg1[128] = {0};
        char arg2[32] = {0};
        char arg3[32] = {0};

        int nargs = sscanf(line, "%31s %127s %31s %31s", cmd, arg1, arg2, arg3);
        if (nargs < 1) continue;

        /* Execute command */
        if (strcmp(cmd, "quit") == 0 || strcmp(cmd, "exit") == 0) {
            break;
        }
        else if (strcmp(cmd, "help") == 0) {
            repl_help();
        }
        else if (strcmp(cmd, "load") == 0) {
            repl_load(&state, arg1);
        }
        else if (strcmp(cmd, "mul") == 0) {
            if (nargs < 3) {
                printf("Usage: mul <u> <v>\n");
            } else {
                xorc_addr_t u = (xorc_addr_t)atoi(arg1);
                xorc_addr_t v = (xorc_addr_t)atoi(arg2);
                repl_mul(&state, u, v);
            }
        }
        else if (strcmp(cmd, "assoc") == 0) {
            if (nargs < 4) {
                printf("Usage: assoc <u> <v> <w>\n");
            } else {
                xorc_addr_t u = (xorc_addr_t)atoi(arg1);
                xorc_addr_t v = (xorc_addr_t)atoi(arg2);
                xorc_addr_t w = (xorc_addr_t)atoi(arg3);
                repl_assoc(&state, u, v, w);
            }
        }
        else if (strcmp(cmd, "comm") == 0) {
            if (nargs < 3) {
                printf("Usage: comm <u> <v>\n");
            } else {
                xorc_addr_t u = (xorc_addr_t)atoi(arg1);
                xorc_addr_t v = (xorc_addr_t)atoi(arg2);
                repl_comm(&state, u, v);
            }
        }
        else if (strcmp(cmd, "table") == 0) {
            repl_table(&state);
        }
        else if (strcmp(cmd, "info") == 0) {
            repl_info(&state);
        }
        else if (strcmp(cmd, "center") == 0) {
            repl_center(&state);
        }
        else if (strcmp(cmd, "radical") == 0) {
            repl_radical(&state);
        }
        else {
            printf("Unknown command: %s (type 'help' for commands)\n", cmd);
        }
    }

    repl_free_state(&state);
    return 0;
}

/* ============================================================================
 * MAIN DRIVER
 * ============================================================================ */

static int compile_source(const char *path)
{
    version();
    printf("Compiling: %s\n", path);

    /* Read source file */
    char *src = read_file(path);
    if (!src) {
        fprintf(stderr, "Error: Cannot read %s\n", path);
        return 1;
    }

    /* Lex */
    lexer_t lex;
    lex_init(&lex, src);

    /* Parse */
    ast_t *ast = parse(&lex);
    if (!ast && src[0] != '\0') {
        /* Parse failed (unless file is empty) */
        /* Check if there were any tokens */
        lexer_t check_lex;
        lex_init(&check_lex, src);
        token_t tok = lex_peek(&check_lex);
        if (tok.kind != TOK_EOF) {
            fprintf(stderr, "Parse error\n");
            free(src);
            return 1;
        }
        /* Empty file - no error, just no output */
    }

    /* Process declarations */
    int errors = 0;
    for (ast_t *node = ast; node; node = node->next) {
        switch (node->kind) {
            case AST_KERNEL:
                process_kernel(node);
                break;
            case AST_VERIFY:
                if (!process_verify(node))
                    errors++;
                break;
            case AST_EXPORT:
                process_export(node);
                break;
        }
    }

    /* Report results */
    if (errors > 0) {
        printf("Result: %d error(s)\n", errors);
    } else {
        printf("Success: 0 errors\n");
    }

    /* Cleanup */
    ast_free(ast);
    kernel_registry_free();
    free_kernel_info();
    free(src);

    return errors > 0 ? 1 : 0;
}

static int inspect_chip(const char *path)
{
    version();
    printf("\n");
    print_chip_info(path);
    return 0;
}

static int verify_chip_seal(const char *path)
{
    version();
    printf("Verifying: %s\n", path);

    bool valid = verify_chip(path);

    return valid ? 0 : 1;
}

int main(int argc, char **argv)
{
    if (argc < 2) {
        usage();
        return 1;
    }

    const char *arg1 = argv[1];

    /* Help */
    if (strcmp(arg1, "-h") == 0 || strcmp(arg1, "--help") == 0) {
        usage();
        return 0;
    }

    /* Version */
    if (strcmp(arg1, "-v") == 0 || strcmp(arg1, "--version") == 0) {
        version();
        return 0;
    }

    /* REPL mode */
    if (strcmp(arg1, "-r") == 0 || strcmp(arg1, "--repl") == 0) {
        return run_repl();
    }

    /* Inspect chip */
    if (strcmp(arg1, "-i") == 0) {
        if (argc < 3) {
            fprintf(stderr, "Error: -i requires a chip file path\n");
            return 1;
        }
        return inspect_chip(argv[2]);
    }

    /* Verify chip seal */
    if (strcmp(arg1, "--verify") == 0) {
        if (argc < 3) {
            fprintf(stderr, "Error: --verify requires a chip file path\n");
            return 1;
        }
        return verify_chip_seal(argv[2]);
    }

    /* Compile source file */
    return compile_source(arg1);
}
