/*
 * XORCE Types - Shared Type Definitions
 *
 * This header defines types shared between parse.c, holo.c, and xorce.c.
 * Centralizes AST definitions, property enums, and related structures.
 *
 * Zero external dependencies (libc only).
 */

#ifndef XORCE_TYPES_H
#define XORCE_TYPES_H

#include "core.h"

/* ============================================================================
 * TOKEN TYPES (must match lex.c)
 * ============================================================================ */

typedef enum {
    TOK_EOF,
    TOK_IDENT,
    TOK_NUMBER,
    TOK_HEX,
    TOK_STRING,
    TOK_KERNEL,
    TOK_VERIFY,
    TOK_EXPORT,
    TOK_AS,
    TOK_FLAT,
    TOK_PAULI,
    TOK_CLIFFORD,
    TOK_CAYLEY,
    TOK_CUSTOM,
    TOK_ASSOCIATIVE,
    TOK_COMMUTATIVE,
    TOK_CENTER,
    TOK_RADICAL,
    TOK_LPAREN,
    TOK_RPAREN,
    TOK_LBRACE,
    TOK_RBRACE,
    TOK_COLON,
    TOK_SEMI,
    TOK_COMMA,
    TOK_EQ,
    TOK_PLUS,
    TOK_MINUS
} token_kind_t;

typedef struct {
    token_kind_t kind;
    union {
        char *str;
        uint32_t num;
    } val;
    int line;
    int col;
} token_t;

typedef struct {
    const char *src;
    int pos;
    int len;
    int line;
    int col;
    token_t current;
    int has_peeked;
} lexer_t;

/* ============================================================================
 * AST TYPES
 * ============================================================================ */

typedef enum {
    AST_KERNEL,
    AST_VERIFY,
    AST_EXPORT
} ast_kind_t;

/* Verification property types */
typedef enum {
    PROP_ASSOCIATIVE,
    PROP_COMMUTATIVE,
    PROP_CENTER,
    PROP_RADICAL
} property_t;

/* Constraint: curvature rectangle for custom kernels */
typedef struct {
    xorc_addr_t a, b, c, d;
    xorc_sign_t sign;
} constraint_t;

/* Kernel specification */
typedef struct {
    char *name;
    xorc_family_t family;
    xorc_dim_t dim;
    union {
        struct { uint8_t p, q; } clifford;
        struct { uint8_t n; } pauli;
        struct { uint8_t level; } cayley;
        struct { constraint_t *list; uint32_t count; uint32_t cap; } custom;
    } params;
} kernel_spec_t;

/* AST node */
typedef struct ast_node {
    ast_kind_t kind;
    int line;  /* Source line number for error reporting */
    union {
        struct {
            char *name;
            kernel_spec_t spec;
        } kernel;
        struct {
            char *name;
            property_t prop;
            uint32_t value;     /* For center=N or radical=N */
        } verify;
        struct {
            char *name;
            char *path;
        } export;
    } data;
    struct ast_node *next;
} ast_t;

/* ============================================================================
 * VERIFICATION RESULT (for holo.c)
 * ============================================================================ */

/* Verification result */
typedef struct {
    bool passed;
    const char *message;
    uint32_t expected;
    uint32_t actual;
    int line;  /* Source line number for error reporting */
} verify_result_t;

/* ============================================================================
 * EXTERNAL LEXER FUNCTIONS (from lex.c)
 * ============================================================================ */

extern void lex_init(lexer_t *l, const char *src);
extern token_t lex_next(lexer_t *l);
extern token_t lex_peek(lexer_t *l);
extern const char* token_name(token_kind_t kind);
extern void token_free(token_t *tok);

/* ============================================================================
 * EXTERNAL PARSER FUNCTIONS (from parse.c)
 * ============================================================================ */

extern ast_t* parse(lexer_t *l);
extern void ast_free(ast_t *ast);
extern void ast_print(ast_t *ast);

/* ============================================================================
 * EXTERNAL HOLO FUNCTIONS (from holo.c)
 * ============================================================================ */

extern void kernel_register(const char *name, xorc_omega_t *omega, xorc_holo_t *holo);
extern xorc_omega_t* kernel_lookup(const char *name);
extern xorc_holo_t* kernel_holo(const char *name);
extern void kernel_registry_free(void);
extern xorc_omega_t* build_kernel(kernel_spec_t *spec);
extern int process_kernel_decl(const char *name, kernel_spec_t *spec);
extern int process_verify_stmt(const char *name, property_t prop, uint32_t value);
extern verify_result_t verify_kernel(const char *name, property_t prop, uint32_t value);

/* ============================================================================
 * EXTERNAL CODEGEN FUNCTIONS (from codegen.c)
 * ============================================================================ */

extern bool codegen_chip(const char *path, xorc_omega_t *omega, xorc_holo_t *holo,
                         xorc_family_t family, void *params);
extern xorc_omega_t* load_chip(const char *path);
extern bool verify_chip(const char *path);
extern void print_chip_info(const char *path);
extern bool chip_info(const char *path, xorc_dim_t *dim, xorc_family_t *family,
                      uint32_t *assoc_fail, uint32_t *comm_fail);

#endif /* XORCE_TYPES_H */
