/*
 * XORCE Parser - Implementation
 *
 * Recursive descent parser per section VII.2.
 * Parses XORCE source language into AST.
 * Zero external dependencies (libc only).
 */

#define _POSIX_C_SOURCE 200809L

#include "types.h"
#include <stdlib.h>
#include <string.h>
#include <stdio.h>

/* ============================================================================
 * PARSER STATE
 * ============================================================================ */

typedef struct {
    lexer_t *lexer;
    token_t tok;
    int error;
} parser_t;

/* ============================================================================
 * PARSER HELPERS
 * ============================================================================ */

static void parser_init(parser_t *p, lexer_t *l)
{
    p->lexer = l;
    p->tok = lex_next(l);
    p->error = 0;
}

static void parser_error(parser_t *p, const char *msg)
{
    fprintf(stderr, "error:%d:%d: %s (got '%s')\n",
            p->tok.line, p->tok.col, msg, token_name(p->tok.kind));
    p->error = 1;
}

static int parser_check(parser_t *p, token_kind_t kind)
{
    return p->tok.kind == kind;
}

static void parser_advance(parser_t *p)
{
    if (p->tok.kind == TOK_IDENT || p->tok.kind == TOK_STRING) {
        /* Don't free - the value is transferred to AST */
    }
    p->tok = lex_next(p->lexer);
}

static int parser_expect(parser_t *p, token_kind_t kind)
{
    if (!parser_check(p, kind)) {
        char msg[64];
        snprintf(msg, sizeof(msg), "expected '%s'", token_name(kind));
        parser_error(p, msg);
        return 0;
    }
    parser_advance(p);
    return 1;
}

/* Consume identifier and return it (caller owns the string) */
static char* parser_consume_ident(parser_t *p)
{
    if (!parser_check(p, TOK_IDENT)) {
        parser_error(p, "expected identifier");
        return NULL;
    }
    char *name = p->tok.val.str;
    p->tok.val.str = NULL;
    parser_advance(p);
    return name;
}

/* Consume string and return it (caller owns the string) */
static char* parser_consume_string(parser_t *p)
{
    if (!parser_check(p, TOK_STRING)) {
        parser_error(p, "expected string");
        return NULL;
    }
    char *str = p->tok.val.str;
    p->tok.val.str = NULL;
    parser_advance(p);
    return str;
}

/* Consume number and return its value */
static uint32_t parser_consume_number(parser_t *p)
{
    if (parser_check(p, TOK_NUMBER)) {
        uint32_t val = p->tok.val.num;
        parser_advance(p);
        return val;
    } else if (parser_check(p, TOK_HEX)) {
        uint32_t val = p->tok.val.num;
        parser_advance(p);
        return val;
    }
    parser_error(p, "expected number");
    return 0;
}

/* ============================================================================
 * AST CONSTRUCTORS
 * ============================================================================ */

static ast_t* ast_alloc(ast_kind_t kind, int line)
{
    ast_t *node = (ast_t*)calloc(1, sizeof(ast_t));
    if (node) {
        node->kind = kind;
        node->line = line;
        node->next = NULL;
    }
    return node;
}

void ast_free(ast_t *ast)
{
    while (ast) {
        ast_t *next = ast->next;
        switch (ast->kind) {
            case AST_KERNEL:
                free(ast->data.kernel.name);
                free(ast->data.kernel.spec.name);
                if (ast->data.kernel.spec.family == XORC_CUSTOM) {
                    free(ast->data.kernel.spec.params.custom.list);
                }
                break;
            case AST_VERIFY:
                free(ast->data.verify.name);
                break;
            case AST_EXPORT:
                free(ast->data.export.name);
                free(ast->data.export.path);
                break;
        }
        free(ast);
        ast = next;
    }
}

/* ============================================================================
 * RECURSIVE DESCENT PARSER
 * ============================================================================ */

/*
 * constraint := addr ',' addr ',' addr ',' addr '=' sign ';'
 */
static int parse_constraint(parser_t *p, constraint_t *c)
{
    c->a = parser_consume_number(p);
    if (p->error) return 0;

    if (!parser_expect(p, TOK_COMMA)) return 0;

    c->b = parser_consume_number(p);
    if (p->error) return 0;

    if (!parser_expect(p, TOK_COMMA)) return 0;

    c->c = parser_consume_number(p);
    if (p->error) return 0;

    if (!parser_expect(p, TOK_COMMA)) return 0;

    c->d = parser_consume_number(p);
    if (p->error) return 0;

    if (!parser_expect(p, TOK_EQ)) return 0;

    if (parser_check(p, TOK_PLUS)) {
        c->sign = 1;
        parser_advance(p);
    } else if (parser_check(p, TOK_MINUS)) {
        c->sign = -1;
        parser_advance(p);
    } else {
        parser_error(p, "expected '+' or '-'");
        return 0;
    }

    if (!parser_expect(p, TOK_SEMI)) return 0;

    return 1;
}

/*
 * kernel_spec := 'flat' dim
 *              | 'pauli' '(' qubits ')'
 *              | 'clifford' '(' p ',' q ')'
 *              | 'cayley' '(' level ')'
 *              | 'custom' '{' constraint* '}'
 */
static int parse_kernel_spec(parser_t *p, kernel_spec_t *spec)
{
    if (parser_check(p, TOK_FLAT)) {
        parser_advance(p);
        spec->family = XORC_FLAT;
        spec->dim = (xorc_dim_t)parser_consume_number(p);
        return !p->error;
    }

    if (parser_check(p, TOK_PAULI)) {
        parser_advance(p);
        spec->family = XORC_PAULI;
        if (!parser_expect(p, TOK_LPAREN)) return 0;
        uint32_t qubits = parser_consume_number(p);
        if (p->error) return 0;
        if (!parser_expect(p, TOK_RPAREN)) return 0;
        spec->params.pauli.n = (uint8_t)qubits;
        spec->dim = (xorc_dim_t)(qubits * 2);
        return 1;
    }

    if (parser_check(p, TOK_CLIFFORD)) {
        parser_advance(p);
        spec->family = XORC_CLIFFORD;
        if (!parser_expect(p, TOK_LPAREN)) return 0;
        uint32_t p_val = parser_consume_number(p);
        if (p->error) return 0;
        if (!parser_expect(p, TOK_COMMA)) return 0;
        uint32_t q_val = parser_consume_number(p);
        if (p->error) return 0;
        if (!parser_expect(p, TOK_RPAREN)) return 0;
        spec->params.clifford.p = (uint8_t)p_val;
        spec->params.clifford.q = (uint8_t)q_val;
        spec->dim = (xorc_dim_t)(p_val + q_val);
        return 1;
    }

    if (parser_check(p, TOK_CAYLEY)) {
        parser_advance(p);
        spec->family = XORC_CAYLEY;
        if (!parser_expect(p, TOK_LPAREN)) return 0;
        uint32_t level = parser_consume_number(p);
        if (p->error) return 0;
        if (!parser_expect(p, TOK_RPAREN)) return 0;
        spec->params.cayley.level = (uint8_t)level;
        spec->dim = (xorc_dim_t)level;
        return 1;
    }

    if (parser_check(p, TOK_CUSTOM)) {
        parser_advance(p);
        spec->family = XORC_CUSTOM;
        spec->dim = 0;  /* Will be determined later */
        spec->params.custom.list = NULL;
        spec->params.custom.count = 0;
        spec->params.custom.cap = 0;

        if (!parser_expect(p, TOK_LBRACE)) return 0;

        /* Parse constraints */
        while (!parser_check(p, TOK_RBRACE) && !parser_check(p, TOK_EOF)) {
            /* Grow constraint list if needed */
            if (spec->params.custom.count >= spec->params.custom.cap) {
                uint32_t new_cap = spec->params.custom.cap == 0 ? 16 : spec->params.custom.cap * 2;
                constraint_t *new_list = (constraint_t*)realloc(
                    spec->params.custom.list,
                    new_cap * sizeof(constraint_t)
                );
                if (!new_list) {
                    parser_error(p, "out of memory");
                    return 0;
                }
                spec->params.custom.list = new_list;
                spec->params.custom.cap = new_cap;
            }

            constraint_t c;
            if (!parse_constraint(p, &c)) return 0;
            spec->params.custom.list[spec->params.custom.count++] = c;

            /* Update dim based on largest address */
            xorc_addr_t max_addr = c.a > c.b ? c.a : c.b;
            max_addr = max_addr > c.c ? max_addr : c.c;
            max_addr = max_addr > c.d ? max_addr : c.d;
            /* Find minimum dimension that accommodates max_addr */
            xorc_dim_t needed_dim = 0;
            while (((uint32_t)1 << needed_dim) <= max_addr) {
                needed_dim++;
            }
            if (needed_dim > spec->dim) {
                spec->dim = needed_dim;
            }
        }

        if (!parser_expect(p, TOK_RBRACE)) return 0;
        return 1;
    }

    parser_error(p, "expected kernel family (flat, pauli, clifford, cayley, custom)");
    return 0;
}

/*
 * kernel_decl := 'kernel' IDENT ':' kernel_spec ';'
 */
static ast_t* parse_kernel_decl(parser_t *p)
{
    int line = p->tok.line;
    parser_advance(p); /* Skip 'kernel' */

    ast_t *node = ast_alloc(AST_KERNEL, line);
    if (!node) {
        parser_error(p, "out of memory");
        return NULL;
    }

    node->data.kernel.name = parser_consume_ident(p);
    if (!node->data.kernel.name) {
        ast_free(node);
        return NULL;
    }

    /* Duplicate name for spec */
    node->data.kernel.spec.name = strdup(node->data.kernel.name);

    if (!parser_expect(p, TOK_COLON)) {
        ast_free(node);
        return NULL;
    }

    if (!parse_kernel_spec(p, &node->data.kernel.spec)) {
        ast_free(node);
        return NULL;
    }

    if (!parser_expect(p, TOK_SEMI)) {
        ast_free(node);
        return NULL;
    }

    return node;
}

/*
 * property := 'associative'
 *           | 'commutative'
 *           | 'center' '=' NUMBER
 *           | 'radical' '=' NUMBER
 */
static int parse_property(parser_t *p, property_t *prop, uint32_t *value)
{
    *value = 0;

    if (parser_check(p, TOK_ASSOCIATIVE)) {
        parser_advance(p);
        *prop = PROP_ASSOCIATIVE;
        return 1;
    }

    if (parser_check(p, TOK_COMMUTATIVE)) {
        parser_advance(p);
        *prop = PROP_COMMUTATIVE;
        return 1;
    }

    if (parser_check(p, TOK_CENTER)) {
        parser_advance(p);
        *prop = PROP_CENTER;
        if (!parser_expect(p, TOK_EQ)) return 0;
        *value = parser_consume_number(p);
        return !p->error;
    }

    if (parser_check(p, TOK_RADICAL)) {
        parser_advance(p);
        *prop = PROP_RADICAL;
        if (!parser_expect(p, TOK_EQ)) return 0;
        *value = parser_consume_number(p);
        return !p->error;
    }

    parser_error(p, "expected property (associative, commutative, center, radical)");
    return 0;
}

/*
 * verify_stmt := 'verify' IDENT ':' property ';'
 */
static ast_t* parse_verify_stmt(parser_t *p)
{
    int line = p->tok.line;
    parser_advance(p); /* Skip 'verify' */

    ast_t *node = ast_alloc(AST_VERIFY, line);
    if (!node) {
        parser_error(p, "out of memory");
        return NULL;
    }

    node->data.verify.name = parser_consume_ident(p);
    if (!node->data.verify.name) {
        ast_free(node);
        return NULL;
    }

    if (!parser_expect(p, TOK_COLON)) {
        ast_free(node);
        return NULL;
    }

    property_t prop;
    uint32_t value;
    if (!parse_property(p, &prop, &value)) {
        ast_free(node);
        return NULL;
    }
    node->data.verify.prop = prop;
    node->data.verify.value = value;

    if (!parser_expect(p, TOK_SEMI)) {
        ast_free(node);
        return NULL;
    }

    return node;
}

/*
 * export_stmt := 'export' IDENT 'as' STRING ';'
 */
static ast_t* parse_export_stmt(parser_t *p)
{
    int line = p->tok.line;
    parser_advance(p); /* Skip 'export' */

    ast_t *node = ast_alloc(AST_EXPORT, line);
    if (!node) {
        parser_error(p, "out of memory");
        return NULL;
    }

    node->data.export.name = parser_consume_ident(p);
    if (!node->data.export.name) {
        ast_free(node);
        return NULL;
    }

    if (!parser_expect(p, TOK_AS)) {
        ast_free(node);
        return NULL;
    }

    node->data.export.path = parser_consume_string(p);
    if (!node->data.export.path) {
        ast_free(node);
        return NULL;
    }

    if (!parser_expect(p, TOK_SEMI)) {
        ast_free(node);
        return NULL;
    }

    return node;
}

/*
 * decl := kernel_decl | verify_stmt | export_stmt
 */
static ast_t* parse_decl(parser_t *p)
{
    if (parser_check(p, TOK_KERNEL)) {
        return parse_kernel_decl(p);
    }
    if (parser_check(p, TOK_VERIFY)) {
        return parse_verify_stmt(p);
    }
    if (parser_check(p, TOK_EXPORT)) {
        return parse_export_stmt(p);
    }

    parser_error(p, "expected declaration (kernel, verify, or export)");
    return NULL;
}

/*
 * program := decl*
 */
ast_t* parse(lexer_t *l)
{
    parser_t p;
    parser_init(&p, l);

    ast_t *head = NULL;
    ast_t *tail = NULL;

    while (!parser_check(&p, TOK_EOF)) {
        ast_t *node = parse_decl(&p);
        if (!node) {
            ast_free(head);
            return NULL;
        }

        if (!head) {
            head = node;
            tail = node;
        } else {
            tail->next = node;
            tail = node;
        }
    }

    return head;
}

/* ============================================================================
 * DEBUG HELPERS
 * ============================================================================ */

static const char* family_name(xorc_family_t family)
{
    switch (family) {
        case XORC_FLAT:     return "flat";
        case XORC_PAULI:    return "pauli";
        case XORC_CLIFFORD: return "clifford";
        case XORC_CAYLEY:   return "cayley";
        case XORC_CUSTOM:   return "custom";
        default:            return "?";
    }
}

static const char* property_name(property_t prop)
{
    switch (prop) {
        case PROP_ASSOCIATIVE: return "associative";
        case PROP_COMMUTATIVE: return "commutative";
        case PROP_CENTER:      return "center";
        case PROP_RADICAL:     return "radical";
        default:               return "?";
    }
}

void ast_print(ast_t *ast)
{
    for (ast_t *node = ast; node; node = node->next) {
        switch (node->kind) {
            case AST_KERNEL: {
                kernel_spec_t *spec = &node->data.kernel.spec;
                printf("KERNEL %s: %s(dim=%d", node->data.kernel.name,
                       family_name(spec->family), spec->dim);
                if (spec->family == XORC_PAULI) {
                    printf(", qubits=%d", spec->params.pauli.n);
                } else if (spec->family == XORC_CLIFFORD) {
                    printf(", p=%d, q=%d", spec->params.clifford.p, spec->params.clifford.q);
                } else if (spec->family == XORC_CAYLEY) {
                    printf(", level=%d", spec->params.cayley.level);
                } else if (spec->family == XORC_CUSTOM) {
                    printf(", constraints=%u", spec->params.custom.count);
                }
                printf(")\n");
                break;
            }
            case AST_VERIFY:
                printf("VERIFY %s: %s", node->data.verify.name,
                       property_name(node->data.verify.prop));
                if (node->data.verify.prop == PROP_CENTER ||
                    node->data.verify.prop == PROP_RADICAL) {
                    printf("=%u", node->data.verify.value);
                }
                printf("\n");
                break;
            case AST_EXPORT:
                printf("EXPORT %s as \"%s\"\n",
                       node->data.export.name, node->data.export.path);
                break;
        }
    }
}
