/*
 * XORCE Holonomy Engine - Implementation
 *
 * Pass 5: Bridge between AST and Core Engine.
 * Builds kernels from parsed specifications.
 * Executes verification predicates.
 * Manages kernel registry for named lookup.
 *
 * Zero external dependencies (libc only).
 *
 * MEMORY OWNERSHIP:
 * - build_kernel(): Allocates and returns xorc_omega_t*. Caller must free with omega_free().
 * - holo_compute(): Allocates and returns xorc_holo_t*. Caller must free with holo_free().
 * - kernel_register(): Borrows name (makes copy), takes ownership of omega and holo.
 * - kernel_lookup(): Returns borrowed pointer. Caller must NOT free.
 * - kernel_holo(): Returns borrowed pointer. Caller must NOT free.
 * - kernel_registry_free(): Frees all registered kernels and holos.
 * - verify_kernel(): Returns stack-allocated result. No cleanup needed.
 */

#define _POSIX_C_SOURCE 200809L

#include "types.h"
#include <stdlib.h>
#include <string.h>
#include <stdio.h>

/* ============================================================================
 * KERNEL REGISTRY
 * ============================================================================ */

#define MAX_KERNELS 64

typedef struct {
    char *name;
    xorc_omega_t *omega;
    xorc_holo_t *holo;
} kernel_entry_t;

static kernel_entry_t registry[MAX_KERNELS];
static int registry_count = 0;

/*
 * Register a named kernel in the registry.
 * Takes ownership of the name string (must be malloc'd).
 */
void kernel_register(const char *name, xorc_omega_t *omega, xorc_holo_t *holo)
{
    if (registry_count >= MAX_KERNELS) {
        fprintf(stderr, "error: kernel registry full (max %d)\n", MAX_KERNELS);
        return;
    }

    /* Check for duplicate */
    for (int i = 0; i < registry_count; i++) {
        if (strcmp(registry[i].name, name) == 0) {
            fprintf(stderr, "error: kernel '%s' already registered\n", name);
            return;
        }
    }

    registry[registry_count].name = strdup(name);
    registry[registry_count].omega = omega;
    registry[registry_count].holo = holo;
    registry_count++;
}

/*
 * Look up a kernel by name.
 * Returns NULL if not found.
 */
xorc_omega_t* kernel_lookup(const char *name)
{
    for (int i = 0; i < registry_count; i++) {
        if (strcmp(registry[i].name, name) == 0) {
            return registry[i].omega;
        }
    }
    return NULL;
}

/*
 * Look up holonomy profile by kernel name.
 * Returns NULL if not found.
 */
xorc_holo_t* kernel_holo(const char *name)
{
    for (int i = 0; i < registry_count; i++) {
        if (strcmp(registry[i].name, name) == 0) {
            return registry[i].holo;
        }
    }
    return NULL;
}

/*
 * Free all registered kernels.
 */
void kernel_registry_free(void)
{
    for (int i = 0; i < registry_count; i++) {
        free(registry[i].name);
        omega_free(registry[i].omega);
        holo_free(registry[i].holo);
    }
    registry_count = 0;
}

/* ============================================================================
 * CONSTRAINT APPLICATION FOR CUSTOM KERNELS
 * ============================================================================ */

/*
 * Compute curvature rectangle value.
 * Curvature(a,b,c,d) = omega(a,b) * omega(a^b, c^d) * omega(c,d) * omega(a^c, b^d)
 *
 * This measures the holonomy around the rectangle formed by
 * the four corners a, b, c, d in the XOR lattice.
 */
static xorc_sign_t curvature_rect(xorc_omega_t *omega, xorc_addr_t a, xorc_addr_t b,
                                   xorc_addr_t c, xorc_addr_t d)
{
    xorc_sign_t s1 = omega_get(omega, a, b);
    xorc_sign_t s2 = omega_get(omega, a ^ b, c ^ d);
    xorc_sign_t s3 = omega_get(omega, c, d);
    xorc_sign_t s4 = omega_get(omega, a ^ c, b ^ d);
    return s1 * s2 * s3 * s4;
}

/*
 * Apply constraint to make curvature rectangle have specified sign.
 * This may require flipping some omega values.
 *
 * Strategy: If current curvature doesn't match desired sign,
 * flip omega(a, b) to change the overall sign.
 */
static void apply_single_constraint(xorc_omega_t *omega, constraint_t *c)
{
    xorc_sign_t current = curvature_rect(omega, c->a, c->b, c->c, c->d);
    if (current != c->sign) {
        /* Flip omega(a, b) to change the curvature sign */
        xorc_sign_t old_val = omega_get(omega, c->a, c->b);
        omega_set(omega, c->a, c->b, -old_val);
    }
}

/*
 * Apply all constraints to a custom kernel.
 * Starts from a flat kernel and applies each constraint in order.
 */
static void apply_constraints(xorc_omega_t *omega, constraint_t *list, uint32_t count)
{
    for (uint32_t i = 0; i < count; i++) {
        apply_single_constraint(omega, &list[i]);
    }
}

/* ============================================================================
 * KERNEL BUILDING
 * ============================================================================ */

/*
 * Build a kernel from an AST specification.
 * Returns a fully populated omega kernel, or NULL on error.
 */
xorc_omega_t* build_kernel(kernel_spec_t *spec)
{
    if (!spec) {
        fprintf(stderr, "error: NULL kernel specification\n");
        return NULL;
    }

    /* Validate dimension */
    if (spec->dim > 16) {
        fprintf(stderr, "error: kernel dimension %d too large (max 16)\n", spec->dim);
        return NULL;
    }

    /* Create omega with specified dimension */
    xorc_omega_t *omega = omega_create(spec->dim);
    if (!omega) {
        fprintf(stderr, "error: failed to allocate omega kernel (dim=%d)\n", spec->dim);
        return NULL;
    }

    /* Initialize based on family */
    switch (spec->family) {
        case XORC_FLAT:
            /* Already initialized to flat by omega_create */
            omega_init_flat(omega);
            break;

        case XORC_PAULI:
            if (spec->dim != spec->params.pauli.n * 2) {
                fprintf(stderr, "warning: pauli dimension mismatch (dim=%d, qubits=%d)\n",
                        spec->dim, spec->params.pauli.n);
            }
            omega_init_pauli(omega, spec->params.pauli.n);
            break;

        case XORC_CLIFFORD:
            if (spec->dim != spec->params.clifford.p + spec->params.clifford.q) {
                fprintf(stderr, "warning: clifford dimension mismatch (dim=%d, p=%d, q=%d)\n",
                        spec->dim, spec->params.clifford.p, spec->params.clifford.q);
            }
            omega_init_clifford(omega, spec->params.clifford.p, spec->params.clifford.q);
            break;

        case XORC_CAYLEY:
            if (spec->dim != spec->params.cayley.level) {
                fprintf(stderr, "warning: cayley dimension mismatch (dim=%d, level=%d)\n",
                        spec->dim, spec->params.cayley.level);
            }
            omega_init_cayley(omega, spec->params.cayley.level);
            break;

        case XORC_CUSTOM:
            /* Start flat, then apply constraints */
            omega_init_flat(omega);
            if (spec->params.custom.list && spec->params.custom.count > 0) {
                apply_constraints(omega, spec->params.custom.list, spec->params.custom.count);
            }
            break;

        default:
            fprintf(stderr, "error: unknown kernel family %d\n", spec->family);
            omega_free(omega);
            return NULL;
    }

    return omega;
}

/* ============================================================================
 * VERIFICATION
 * ============================================================================ */

/*
 * Property name for error messages.
 */
static const char* property_name(property_t prop)
{
    switch (prop) {
        case PROP_ASSOCIATIVE: return "associative";
        case PROP_COMMUTATIVE: return "commutative";
        case PROP_CENTER:      return "center";
        case PROP_RADICAL:     return "radical";
        default:               return "unknown";
    }
}

/*
 * Verify a property on a named kernel.
 */
verify_result_t verify_kernel(const char *name, property_t prop, uint32_t value)
{
    verify_result_t result = { false, "unknown error", 0, 0, 0 };

    /* Look up the kernel */
    xorc_omega_t *omega = kernel_lookup(name);
    if (!omega) {
        result.message = "kernel not found";
        return result;
    }

    xorc_holo_t *holo = kernel_holo(name);
    if (!holo) {
        result.message = "holonomy profile not computed";
        return result;
    }

    switch (prop) {
        case PROP_ASSOCIATIVE:
            result.expected = 0;
            result.actual = holo->assoc_failures;
            if (holo_is_associative(holo)) {
                result.passed = true;
                result.message = "is associative";
            } else {
                result.passed = false;
                result.message = "not associative";
            }
            break;

        case PROP_COMMUTATIVE:
            result.expected = 0;
            result.actual = holo->comm_failures;
            if (holo_is_commutative(holo)) {
                result.passed = true;
                result.message = "is commutative";
            } else {
                result.passed = false;
                result.message = "not commutative";
            }
            break;

        case PROP_CENTER:
            result.expected = value;
            result.actual = holo_center_dim(omega);
            if (result.actual == value) {
                result.passed = true;
                result.message = "center dimension matches";
            } else {
                result.passed = false;
                result.message = "center dimension mismatch";
            }
            break;

        case PROP_RADICAL:
            result.expected = value;
            result.actual = holo_radical_dim(omega);
            if (result.actual == value) {
                result.passed = true;
                result.message = "radical dimension matches";
            } else {
                result.passed = false;
                result.message = "radical dimension mismatch";
            }
            break;

        default:
            result.message = "unknown property";
            break;
    }

    return result;
}

/* ============================================================================
 * HIGH-LEVEL PROCESSING FUNCTIONS
 * ============================================================================ */

/*
 * Process a kernel declaration from AST.
 * Builds the kernel, computes holonomy, and registers it.
 * Returns 0 on success, -1 on error.
 */
int process_kernel_decl(const char *name, kernel_spec_t *spec)
{
    /* Build the kernel */
    xorc_omega_t *omega = build_kernel(spec);
    if (!omega) {
        fprintf(stderr, "error: failed to build kernel '%s'\n", name);
        return -1;
    }

    /* Compute holonomy profile */
    xorc_holo_t *holo = holo_compute(omega);
    if (!holo) {
        fprintf(stderr, "error: failed to compute holonomy for kernel '%s'\n", name);
        omega_free(omega);
        return -1;
    }

    /* Register the kernel */
    kernel_register(name, omega, holo);

    /* Report kernel info */
    fprintf(stdout, "kernel '%s': dim=%d, size=%u, assoc_failures=%u, comm_failures=%u\n",
            name, omega->dim, omega->size, holo->assoc_failures, holo->comm_failures);

    return 0;
}

/*
 * Process a verify statement from AST.
 * Returns 0 on pass, -1 on fail or error.
 */
int process_verify_stmt(const char *name, property_t prop, uint32_t value)
{
    verify_result_t result = verify_kernel(name, prop, value);

    if (result.passed) {
        fprintf(stdout, "verify '%s' : %s => PASS (%s)\n",
                name, property_name(prop), result.message);
        return 0;
    } else {
        if (prop == PROP_CENTER || prop == PROP_RADICAL) {
            fprintf(stderr, "verify '%s' : %s=%u => FAIL (expected %u, got %u)\n",
                    name, property_name(prop), value, result.expected, result.actual);
        } else {
            fprintf(stderr, "verify '%s' : %s => FAIL (%s, failures=%u)\n",
                    name, property_name(prop), result.message, result.actual);
        }
        return -1;
    }
}

/* ============================================================================
 * HOLONOMY INSPECTION UTILITIES
 * ============================================================================ */

/*
 * Print detailed holonomy information for a kernel.
 */
void holo_print_info(const char *name)
{
    xorc_omega_t *omega = kernel_lookup(name);
    xorc_holo_t *holo = kernel_holo(name);

    if (!omega || !holo) {
        fprintf(stderr, "error: kernel '%s' not found\n", name);
        return;
    }

    printf("=== Holonomy Profile: %s ===\n", name);
    printf("Dimension:           %d\n", omega->dim);
    printf("Size (|V|):          %u\n", omega->size);
    printf("Associator failures: %u\n", holo->assoc_failures);
    printf("Commutator failures: %u\n", holo->comm_failures);
    printf("Is associative:      %s\n", holo_is_associative(holo) ? "yes" : "no");
    printf("Is commutative:      %s\n", holo_is_commutative(holo) ? "yes" : "no");
    printf("Center dimension:    %u\n", holo_center_dim(omega));
    printf("Radical dimension:   %u\n", holo_radical_dim(omega));
    printf("========================\n");
}

/*
 * Print the omega table for a kernel (for debugging).
 * Only useful for small dimensions (dim <= 4).
 */
void holo_print_omega(const char *name)
{
    xorc_omega_t *omega = kernel_lookup(name);

    if (!omega) {
        fprintf(stderr, "error: kernel '%s' not found\n", name);
        return;
    }

    if (omega->dim > 4) {
        printf("(omega table too large to display, dim=%d)\n", omega->dim);
        return;
    }

    printf("Omega table for '%s' (dim=%d, size=%u):\n", name, omega->dim, omega->size);
    printf("     ");
    for (uint32_t v = 0; v < omega->size; v++) {
        printf("%3u ", v);
    }
    printf("\n");

    for (uint32_t u = 0; u < omega->size; u++) {
        printf("%3u: ", u);
        for (uint32_t v = 0; v < omega->size; v++) {
            xorc_sign_t s = omega_get(omega, u, v);
            printf(" %c  ", s > 0 ? '+' : '-');
        }
        printf("\n");
    }
}

/* ============================================================================
 * KERNEL ANALYSIS UTILITIES
 * ============================================================================ */

/*
 * Find all 3-bubble failures (non-associative triples).
 * Useful for understanding the structure of non-associative algebras.
 * Returns count of failures found (up to max_results).
 */
uint32_t holo_find_assoc_failures(const char *name, uint32_t max_results,
                                   xorc_addr_t *u_out, xorc_addr_t *v_out,
                                   xorc_addr_t *w_out)
{
    xorc_omega_t *omega = kernel_lookup(name);
    if (!omega) return 0;

    uint32_t count = 0;
    for (uint32_t u = 0; u < omega->size && count < max_results; u++) {
        for (uint32_t v = 0; v < omega->size && count < max_results; v++) {
            for (uint32_t w = 0; w < omega->size && count < max_results; w++) {
                if (bubble3(omega, u, v, w) == -1) {
                    if (u_out) u_out[count] = u;
                    if (v_out) v_out[count] = v;
                    if (w_out) w_out[count] = w;
                    count++;
                }
            }
        }
    }
    return count;
}

/*
 * Find all 2-bubble failures (non-commutative pairs).
 * Returns count of failures found (up to max_results).
 */
uint32_t holo_find_comm_failures(const char *name, uint32_t max_results,
                                  xorc_addr_t *u_out, xorc_addr_t *v_out)
{
    xorc_omega_t *omega = kernel_lookup(name);
    if (!omega) return 0;

    uint32_t count = 0;
    for (uint32_t u = 0; u < omega->size && count < max_results; u++) {
        for (uint32_t v = 0; v < omega->size && count < max_results; v++) {
            if (bubble2(omega, u, v) == -1) {
                if (u_out) u_out[count] = u;
                if (v_out) v_out[count] = v;
                count++;
            }
        }
    }
    return count;
}

/*
 * Get all central elements of a kernel.
 * Returns count of central elements (up to max_results).
 */
uint32_t holo_get_center(const char *name, uint32_t max_results, xorc_addr_t *center_out)
{
    xorc_omega_t *omega = kernel_lookup(name);
    if (!omega) return 0;

    uint32_t count = 0;
    for (uint32_t z = 0; z < omega->size && count < max_results; z++) {
        bool central = true;
        for (uint32_t v = 0; v < omega->size; v++) {
            if (bubble2(omega, z, v) == -1) {
                central = false;
                break;
            }
        }
        if (central) {
            if (center_out) center_out[count] = z;
            count++;
        }
    }
    return count;
}

/*
 * Get all radical elements of a kernel.
 * Returns count of radical elements (up to max_results).
 */
uint32_t holo_get_radical(const char *name, uint32_t max_results, xorc_addr_t *radical_out)
{
    xorc_omega_t *omega = kernel_lookup(name);
    if (!omega) return 0;

    uint32_t count = 0;
    for (uint32_t r = 0; r < omega->size && count < max_results; r++) {
        bool in_radical = true;

        /* Check symmetry: omega(r,v) = omega(v,r) for all v */
        for (uint32_t v = 0; v < omega->size && in_radical; v++) {
            if (omega_get(omega, r, v) != omega_get(omega, v, r)) {
                in_radical = false;
            }
        }

        /* Check associativity: bubble3(r,u,v) = +1 for all u,v */
        for (uint32_t u = 0; u < omega->size && in_radical; u++) {
            for (uint32_t v = 0; v < omega->size && in_radical; v++) {
                if (bubble3(omega, r, u, v) == -1) {
                    in_radical = false;
                }
            }
        }

        if (in_radical) {
            if (radical_out) radical_out[count] = r;
            count++;
        }
    }
    return count;
}
