/*
 * XORCE Core Engine - Header
 *
 * Algebraic foundation for the XOR Algebra Compiler.
 * Implements phase kernels, holonomy computation, and verification.
 *
 * Zero external dependencies (libc only).
 */

#ifndef XORCE_CORE_H
#define XORCE_CORE_H

#include <stdint.h>
#include <stdbool.h>
#include <stddef.h>

/* ============================================================================
 * CORE TYPES (Section V.1)
 * ============================================================================ */

/* Sign: +1 or -1 */
typedef int8_t xorc_sign_t;

/* Address: element of V = F_2^n */
typedef uint32_t xorc_addr_t;

/* Dimension: n where |V| = 2^n */
typedef uint8_t xorc_dim_t;

/* Kernel family */
typedef enum {
    XORC_FLAT,
    XORC_PAULI,
    XORC_CLIFFORD,
    XORC_CAYLEY,
    XORC_CUSTOM
} xorc_family_t;

/* Phase kernel omega: V x V -> {+1, -1} */
typedef struct {
    xorc_dim_t dim;         /* n where |V| = 2^n */
    uint32_t size;          /* 2^dim */
    xorc_sign_t *data;      /* size x size, row-major */
} xorc_omega_t;

/* Holonomy profile */
typedef struct {
    xorc_dim_t dim;
    uint32_t size;
    uint32_t assoc_failures;    /* Count of 3-bubble = -1 */
    uint32_t comm_failures;     /* Count of 2-bubble = -1 */
} xorc_holo_t;

/* ============================================================================
 * OMEGA OPERATIONS
 * ============================================================================ */

/*
 * Create a new omega kernel with given dimension.
 * Allocates size x size table, initialized to +1.
 * Returns NULL on allocation failure.
 */
xorc_omega_t* omega_create(xorc_dim_t dim);

/*
 * Free an omega kernel and its data.
 */
void omega_free(xorc_omega_t *omega);

/*
 * Get omega(u, v).
 */
xorc_sign_t omega_get(xorc_omega_t *omega, xorc_addr_t u, xorc_addr_t v);

/*
 * Set omega(u, v) = s.
 */
void omega_set(xorc_omega_t *omega, xorc_addr_t u, xorc_addr_t v, xorc_sign_t s);

/* ============================================================================
 * KERNEL INITIALIZATION (Section VI.1)
 * ============================================================================ */

/*
 * Initialize flat kernel: omega(u,v) = +1 for all u,v.
 */
void omega_init_flat(xorc_omega_t *omega);

/*
 * Initialize Pauli kernel for n qubits.
 * Dimension of omega must be 2n.
 * omega(u,v) = (-1)^(popcount(x2 & z1)) where:
 *   x2 = lower n bits of v
 *   z1 = upper n bits of u
 */
void omega_init_pauli(xorc_omega_t *omega, uint8_t qubits);

/*
 * Initialize Clifford algebra Cl(p,q).
 * Dimension of omega must be n = p + q.
 * omega(u,v) = (-1)^(tau ^ sigma) where:
 *   tau = XOR over i<j of bit_i(u) & bit_j(v)
 *   sigma = XOR over i in [p,n) of bit_i(u) & bit_i(v)
 */
void omega_init_clifford(xorc_omega_t *omega, uint8_t p, uint8_t q);

/*
 * Initialize Cayley-Dickson algebra at given level.
 * Level 0: reals (flat, dim 0)
 * Level 1: complex (dim 1)
 * Level 2: quaternions (dim 2)
 * Level 3: octonions (dim 3)
 * etc.
 */
void omega_init_cayley(xorc_omega_t *omega, uint8_t level);

/* ============================================================================
 * HOLONOMY COMPUTATION (Section VI.2)
 * ============================================================================ */

/*
 * 3-Bubble (associator holonomy):
 * Omega(u,v,w) = omega(u,v) * omega(u^v,w) * omega(v,w) * omega(u,v^w)
 *
 * Returns +1 if associative at (u,v,w), -1 if not.
 */
xorc_sign_t bubble3(xorc_omega_t *omega, xorc_addr_t u, xorc_addr_t v, xorc_addr_t w);

/*
 * 2-Bubble (commutator holonomy):
 * B(u,v) = omega(u,v) * omega(v,u)
 *
 * Returns +1 if commutative at (u,v), -1 if not.
 */
xorc_sign_t bubble2(xorc_omega_t *omega, xorc_addr_t u, xorc_addr_t v);

/*
 * Compute full holonomy profile.
 * Counts all 3-bubble and 2-bubble failures.
 * Returns NULL on allocation failure.
 */
xorc_holo_t* holo_compute(xorc_omega_t *omega);

/*
 * Free holonomy profile.
 */
void holo_free(xorc_holo_t *holo);

/* ============================================================================
 * VERIFICATION PREDICATES (Section VI.3)
 * ============================================================================ */

/*
 * Check if algebra is associative (all 3-bubbles = +1).
 */
bool holo_is_associative(xorc_holo_t *holo);

/*
 * Check if algebra is commutative (all 2-bubbles = +1).
 */
bool holo_is_commutative(xorc_holo_t *holo);

/*
 * Compute dimension of center.
 * Center = {z : B(z,v) = +1 for all v}.
 * Returns count of central elements.
 */
uint32_t holo_center_dim(xorc_omega_t *omega);

/*
 * Compute dimension of radical.
 * Radical = {r : omega(r,v) = omega(v,r) for all v and bubble3(r,u,v) = +1 for all u,v}.
 * Returns count of radical elements.
 */
uint32_t holo_radical_dim(xorc_omega_t *omega);

/* ============================================================================
 * ALGEBRA ELEMENT OPERATIONS (Section VI.5)
 * ============================================================================ */

/*
 * Element representation: coefficient * basis element
 * δ_u with coefficient +1 or -1
 */
typedef struct {
    xorc_sign_t coeff;    /* +1 or -1 */
    xorc_addr_t basis;    /* index in F_2^n */
} xorc_elem_t;

/*
 * Multiply two basis elements: δ_u * δ_v = ω(u,v) · δ_{u⊕v}
 * Returns the resulting element.
 */
xorc_elem_t elem_mul(xorc_omega_t *omega, xorc_addr_t u, xorc_addr_t v);

/*
 * Multiply two elements (with coefficients).
 * (a·δ_u) * (b·δ_v) = a·b·ω(u,v) · δ_{u⊕v}
 */
xorc_elem_t elem_mul2(xorc_omega_t *omega, xorc_elem_t a, xorc_elem_t b);

/*
 * Associator: (a*b)*c vs a*(b*c)
 * Returns the sign difference:
 *   +1 if associative at (a,b,c)
 *   -1 if non-associative
 * This is equivalent to bubble3(omega, a, b, c).
 */
xorc_sign_t elem_assoc(xorc_omega_t *omega, xorc_addr_t a, xorc_addr_t b, xorc_addr_t c);

/*
 * Commutator: a*b vs b*a
 * Returns the sign difference:
 *   +1 if commutative at (a,b)
 *   -1 if anticommutative
 * This is equivalent to bubble2(omega, a, b).
 */
xorc_sign_t elem_comm(xorc_omega_t *omega, xorc_addr_t a, xorc_addr_t b);

/*
 * List center elements.
 * center_list must have at least omega->size entries.
 * Returns the count of center elements.
 */
uint32_t holo_center_list(xorc_omega_t *omega, xorc_addr_t *center_list);

/*
 * List radical elements.
 * radical_list must have at least omega->size entries.
 * Returns the count of radical elements.
 */
uint32_t holo_radical_list(xorc_omega_t *omega, xorc_addr_t *radical_list);

/* ============================================================================
 * SHA-256 SEALING (Section VI.4)
 * ============================================================================ */

/*
 * Compute SHA-256 hash of data.
 * hash must point to 32-byte buffer.
 */
void sha256(const uint8_t *data, size_t len, uint8_t *hash);

/*
 * Compute seal (SHA-256) over data.
 * seal must point to 32-byte buffer.
 */
void seal_compute(const uint8_t *data, size_t len, uint8_t *seal);

/*
 * Verify seal matches data.
 */
bool seal_verify(const uint8_t *data, size_t len, const uint8_t *seal);

#endif /* XORCE_CORE_H */
