/*
 * XORCE Code Generation - Implementation
 *
 * Pass 6: Generate verified, sealed .xorc chip files.
 * Produces the binary output format from omega kernels.
 * Loads and verifies existing chip files.
 *
 * Zero external dependencies (libc only).
 *
 * MEMORY OWNERSHIP:
 * - codegen_chip(): Borrows omega and holo pointers. Caller retains ownership.
 * - load_chip(): Allocates and returns xorc_omega_t*. Caller must free with omega_free().
 * - verify_chip(): Borrows path string. No allocations retained by caller.
 * - print_chip_info(): Borrows path string. No allocations retained by caller.
 * - export_kernel(): Borrows all pointers. No allocations retained by caller.
 * - compare_chips(): Borrows paths. Allocates temp omegas internally (freed before return).
 * - dump_chip_omega(): Borrows path. Allocates temp omega internally (freed before return).
 */

#include "core.h"
#include <stdlib.h>
#include <string.h>
#include <stdio.h>

/* ============================================================================
 * CHIP FILE FORMAT CONSTANTS
 * ============================================================================ */

/* Magic number: "XORC" */
#define XORC_MAGIC_0 'X'
#define XORC_MAGIC_1 'O'
#define XORC_MAGIC_2 'R'
#define XORC_MAGIC_3 'C'

/* Version: 1.0 */
#define XORC_VERSION 0x0100

/* Flags */
#define XORC_FLAG_ASSOC   0x0001   /* Is associative */
#define XORC_FLAG_COMM    0x0002   /* Is commutative */
#define XORC_FLAG_SEALED  0x0004   /* Has valid seal */

/* Header size */
#define XORC_HEADER_SIZE 64

/* ============================================================================
 * CHIP HEADER STRUCTURE (Section V.2)
 *
 * Layout (64 bytes total):
 *   magic[4]      =  4 bytes   offset  0
 *   version[2]    =  2 bytes   offset  4
 *   flags[2]      =  2 bytes   offset  6
 *   dim[1]        =  1 byte    offset  8
 *   family[1]     =  1 byte    offset  9
 *   params[2]     =  2 bytes   offset 10  (family-specific: p,q or qubits or level)
 *   size[4]       =  4 bytes   offset 12
 *   assoc_fail[4] =  4 bytes   offset 16
 *   comm_fail[4]  =  4 bytes   offset 20
 *   omega_off[4]  =  4 bytes   offset 24
 *   omega_sz[4]   =  4 bytes   offset 28
 *   seal[32]      = 32 bytes   offset 32
 *                  --------
 *   Total         = 64 bytes
 * ============================================================================ */

typedef struct {
    uint8_t  magic[4];       /* "XORC" */
    uint16_t version;        /* 0x0100 */
    uint16_t flags;          /* ASSOC|COMM|SEALED */
    uint8_t  dim;            /* n */
    uint8_t  family;         /* 0=FLAT, 1=PAULI, 2=CLIFFORD, 3=CAYLEY, 4=CUSTOM */
    uint8_t  params[2];      /* Family-specific: [p,q] for Clifford, [qubits,0] for Pauli, [level,0] for Cayley */
    uint32_t size;           /* 2^n */
    uint32_t assoc_fail;     /* Count of 3-bubble failures */
    uint32_t comm_fail;      /* Count of 2-bubble failures */
    uint32_t omega_off;      /* Offset to omega table (always 64) */
    uint32_t omega_sz;       /* Size of omega table (size^2) */
    uint8_t  seal[32];       /* SHA-256 of header+data with seal zeroed */
} __attribute__((packed)) xorc_header_t;

/* Compile-time assertion: header must be exactly 64 bytes */
_Static_assert(sizeof(xorc_header_t) == 64, "xorc_header_t must be 64 bytes");

/* ============================================================================
 * FAMILY NAME UTILITIES
 * ============================================================================ */

/*
 * Get human-readable family name.
 */
static const char* family_name(uint8_t family)
{
    switch (family) {
        case XORC_FLAT:     return "FLAT";
        case XORC_PAULI:    return "PAULI";
        case XORC_CLIFFORD: return "CLIFFORD";
        case XORC_CAYLEY:   return "CAYLEY";
        case XORC_CUSTOM:   return "CUSTOM";
        default:            return "UNKNOWN";
    }
}

/* ============================================================================
 * PARAMS ENCODING (Section VII.4)
 * ============================================================================ */

/*
 * Encode family-specific parameters into the params[2] field.
 *
 * FLAT:     params unused (zeros)
 * PAULI:    params[0] = qubits, params[1] = 0
 * CLIFFORD: params[0] = p, params[1] = q
 * CAYLEY:   params[0] = level, params[1] = 0
 * CUSTOM:   params unused (zeros)
 */
typedef struct {
    union {
        struct { uint8_t qubits; } pauli;
        struct { uint8_t p; uint8_t q; } clifford;
        struct { uint8_t level; } cayley;
        uint8_t raw[2];
    };
} codegen_params_t;

static void encode_params(uint8_t *dest, xorc_family_t family, void *params)
{
    memset(dest, 0, 2);

    if (!params) return;

    codegen_params_t *p = (codegen_params_t*)params;

    switch (family) {
        case XORC_FLAT:
            /* No parameters */
            break;

        case XORC_PAULI:
            dest[0] = p->pauli.qubits;
            break;

        case XORC_CLIFFORD:
            dest[0] = p->clifford.p;
            dest[1] = p->clifford.q;
            break;

        case XORC_CAYLEY:
            dest[0] = p->cayley.level;
            break;

        case XORC_CUSTOM:
            /* No parameters stored in chip (constraints applied at build time) */
            break;
    }
}

/* ============================================================================
 * CODEGEN_CHIP: Generate .xorc chip file
 * ============================================================================ */

/*
 * Generate a .xorc chip file from omega kernel and holonomy profile.
 *
 * Steps:
 * 1. Allocate buffer: 64 + omega->size * omega->size bytes
 * 2. Fill header with all metadata
 * 3. Copy omega data after header (row-major int8_t values)
 * 4. Compute seal (SHA-256 with seal field zeroed)
 * 5. Write to file
 *
 * Returns true on success, false on error.
 */
bool codegen_chip(const char *path, xorc_omega_t *omega, xorc_holo_t *holo,
                  xorc_family_t family, void *params)
{
    if (!path || !omega || !holo) {
        fprintf(stderr, "codegen: invalid arguments\n");
        return false;
    }

    /* Calculate sizes */
    uint32_t omega_sz = omega->size * omega->size;
    size_t total_size = XORC_HEADER_SIZE + omega_sz;

    /* Allocate buffer */
    uint8_t *buffer = (uint8_t*)malloc(total_size);
    if (!buffer) {
        fprintf(stderr, "codegen: failed to allocate buffer (%zu bytes)\n", total_size);
        return false;
    }

    /* Clear buffer */
    memset(buffer, 0, total_size);

    /* Fill header */
    xorc_header_t *header = (xorc_header_t*)buffer;

    /* Magic: "XORC" */
    header->magic[0] = XORC_MAGIC_0;
    header->magic[1] = XORC_MAGIC_1;
    header->magic[2] = XORC_MAGIC_2;
    header->magic[3] = XORC_MAGIC_3;

    /* Version: 1.0 */
    header->version = XORC_VERSION;

    /* Flags: start with none, set based on properties */
    header->flags = 0;

    /* Dimension */
    header->dim = omega->dim;

    /* Family */
    header->family = (uint8_t)family;

    /* Family-specific parameters */
    encode_params(header->params, family, params);

    /* Size */
    header->size = omega->size;

    /* Holonomy failure counts */
    header->assoc_fail = holo->assoc_failures;
    header->comm_fail = holo->comm_failures;

    /* Omega table location */
    header->omega_off = XORC_HEADER_SIZE;
    header->omega_sz = omega_sz;

    /* Set flags based on holonomy */
    if (holo->assoc_failures == 0) {
        header->flags |= XORC_FLAG_ASSOC;
    }
    if (holo->comm_failures == 0) {
        header->flags |= XORC_FLAG_COMM;
    }

    /* Copy omega data (row-major, int8_t +1/-1 values) */
    int8_t *omega_data = (int8_t*)(buffer + XORC_HEADER_SIZE);
    for (uint32_t i = 0; i < omega_sz; i++) {
        omega_data[i] = omega->data[i];
    }

    /* Set SEALED flag BEFORE computing seal (so it's included in hash) */
    header->flags |= XORC_FLAG_SEALED;

    /* Compute seal: SHA-256 of entire buffer with seal field zeroed */
    memset(header->seal, 0, 32);
    sha256(buffer, total_size, header->seal);

    /* Write to file */
    FILE *f = fopen(path, "wb");
    if (!f) {
        fprintf(stderr, "codegen: failed to open '%s' for writing\n", path);
        free(buffer);
        return false;
    }

    size_t written = fwrite(buffer, 1, total_size, f);
    fclose(f);

    if (written != total_size) {
        fprintf(stderr, "codegen: failed to write complete chip (wrote %zu of %zu bytes)\n",
                written, total_size);
        free(buffer);
        return false;
    }

    free(buffer);

    printf("codegen: wrote '%s' (%zu bytes, dim=%d, size=%u)\n",
           path, total_size, omega->dim, omega->size);

    return true;
}

/* ============================================================================
 * LOAD_CHIP: Load .xorc chip file
 * ============================================================================ */

/*
 * Load a .xorc chip file and return the omega kernel.
 *
 * Steps:
 * 1. Read header (64 bytes)
 * 2. Verify magic == "XORC"
 * 3. Verify seal if SEALED flag set
 * 4. Allocate omega with header->dim
 * 5. Read omega data
 * 6. Return omega
 *
 * Returns omega kernel on success, NULL on error.
 * Caller is responsible for freeing the returned omega.
 */
xorc_omega_t* load_chip(const char *path)
{
    if (!path) {
        fprintf(stderr, "load_chip: invalid path\n");
        return NULL;
    }

    /* Open file */
    FILE *f = fopen(path, "rb");
    if (!f) {
        fprintf(stderr, "load_chip: failed to open '%s'\n", path);
        return NULL;
    }

    /* Read header */
    xorc_header_t header;
    size_t read_bytes = fread(&header, 1, XORC_HEADER_SIZE, f);
    if (read_bytes != XORC_HEADER_SIZE) {
        fprintf(stderr, "load_chip: failed to read header (got %zu bytes)\n", read_bytes);
        fclose(f);
        return NULL;
    }

    /* Verify magic */
    if (header.magic[0] != XORC_MAGIC_0 ||
        header.magic[1] != XORC_MAGIC_1 ||
        header.magic[2] != XORC_MAGIC_2 ||
        header.magic[3] != XORC_MAGIC_3) {
        fprintf(stderr, "load_chip: invalid magic (expected 'XORC', got '%c%c%c%c')\n",
                header.magic[0], header.magic[1], header.magic[2], header.magic[3]);
        fclose(f);
        return NULL;
    }

    /* Validate dimensions */
    if (header.dim > 16) {
        fprintf(stderr, "load_chip: dimension %d exceeds maximum (16)\n", header.dim);
        fclose(f);
        return NULL;
    }

    uint32_t expected_size = (uint32_t)1 << header.dim;
    if (header.size != expected_size) {
        fprintf(stderr, "load_chip: size mismatch (header=%u, expected=%u for dim=%d)\n",
                header.size, expected_size, header.dim);
        fclose(f);
        return NULL;
    }

    /* Validate omega table parameters */
    if (header.omega_off != XORC_HEADER_SIZE) {
        fprintf(stderr, "load_chip: unexpected omega offset (got %u, expected %d)\n",
                header.omega_off, XORC_HEADER_SIZE);
        fclose(f);
        return NULL;
    }

    uint32_t expected_omega_sz = header.size * header.size;
    if (header.omega_sz != expected_omega_sz) {
        fprintf(stderr, "load_chip: omega size mismatch (got %u, expected %u)\n",
                header.omega_sz, expected_omega_sz);
        fclose(f);
        return NULL;
    }

    /* If SEALED flag is set, verify the seal */
    if (header.flags & XORC_FLAG_SEALED) {
        /* Seek back to start and read entire file */
        fseek(f, 0, SEEK_SET);
        size_t total_size = XORC_HEADER_SIZE + header.omega_sz;
        uint8_t *buffer = (uint8_t*)malloc(total_size);
        if (!buffer) {
            fprintf(stderr, "load_chip: failed to allocate verification buffer\n");
            fclose(f);
            return NULL;
        }

        read_bytes = fread(buffer, 1, total_size, f);
        if (read_bytes != total_size) {
            fprintf(stderr, "load_chip: failed to read file for verification\n");
            free(buffer);
            fclose(f);
            return NULL;
        }

        /* Save the seal, zero it, compute hash, compare */
        uint8_t saved_seal[32];
        memcpy(saved_seal, buffer + offsetof(xorc_header_t, seal), 32);
        memset(buffer + offsetof(xorc_header_t, seal), 0, 32);

        uint8_t computed_seal[32];
        sha256(buffer, total_size, computed_seal);

        if (memcmp(saved_seal, computed_seal, 32) != 0) {
            fprintf(stderr, "load_chip: seal verification FAILED\n");
            free(buffer);
            fclose(f);
            return NULL;
        }

        free(buffer);

        /* Seek to omega data position */
        fseek(f, XORC_HEADER_SIZE, SEEK_SET);
    }

    /* Create omega kernel */
    xorc_omega_t *omega = omega_create(header.dim);
    if (!omega) {
        fprintf(stderr, "load_chip: failed to allocate omega kernel\n");
        fclose(f);
        return NULL;
    }

    /* Read omega data */
    read_bytes = fread(omega->data, 1, header.omega_sz, f);
    if (read_bytes != header.omega_sz) {
        fprintf(stderr, "load_chip: failed to read omega data (got %zu of %u bytes)\n",
                read_bytes, header.omega_sz);
        omega_free(omega);
        fclose(f);
        return NULL;
    }

    fclose(f);

    return omega;
}

/* ============================================================================
 * VERIFY_CHIP: Verify chip seal
 * ============================================================================ */

/*
 * Verify the seal of a .xorc chip file.
 *
 * Steps:
 * 1. Read entire file
 * 2. Save seal, zero seal field
 * 3. Compute SHA-256
 * 4. Compare with saved seal
 * 5. Report result
 *
 * Returns true if seal is valid, false otherwise.
 */
bool verify_chip(const char *path)
{
    if (!path) {
        fprintf(stderr, "verify_chip: invalid path\n");
        return false;
    }

    /* Open file */
    FILE *f = fopen(path, "rb");
    if (!f) {
        fprintf(stderr, "verify_chip: failed to open '%s'\n", path);
        return false;
    }

    /* Get file size */
    fseek(f, 0, SEEK_END);
    long file_size = ftell(f);
    fseek(f, 0, SEEK_SET);

    if (file_size < XORC_HEADER_SIZE) {
        fprintf(stderr, "verify_chip: file too small (%ld bytes, minimum %d)\n",
                file_size, XORC_HEADER_SIZE);
        fclose(f);
        return false;
    }

    /* Read entire file */
    uint8_t *buffer = (uint8_t*)malloc((size_t)file_size);
    if (!buffer) {
        fprintf(stderr, "verify_chip: failed to allocate buffer\n");
        fclose(f);
        return false;
    }

    size_t read_bytes = fread(buffer, 1, (size_t)file_size, f);
    fclose(f);

    if (read_bytes != (size_t)file_size) {
        fprintf(stderr, "verify_chip: failed to read file\n");
        free(buffer);
        return false;
    }

    /* Verify magic */
    xorc_header_t *header = (xorc_header_t*)buffer;
    if (header->magic[0] != XORC_MAGIC_0 ||
        header->magic[1] != XORC_MAGIC_1 ||
        header->magic[2] != XORC_MAGIC_2 ||
        header->magic[3] != XORC_MAGIC_3) {
        fprintf(stderr, "verify_chip: invalid magic\n");
        free(buffer);
        return false;
    }

    /* Check if SEALED flag is set */
    if (!(header->flags & XORC_FLAG_SEALED)) {
        fprintf(stderr, "verify_chip: chip is not sealed\n");
        free(buffer);
        return false;
    }

    /* Save the seal */
    uint8_t saved_seal[32];
    memcpy(saved_seal, header->seal, 32);

    /* Zero the seal field */
    memset(header->seal, 0, 32);

    /* Compute SHA-256 */
    uint8_t computed_seal[32];
    sha256(buffer, (size_t)file_size, computed_seal);

    /* Compare */
    bool valid = (memcmp(saved_seal, computed_seal, 32) == 0);

    free(buffer);

    if (valid) {
        printf("verify_chip: '%s' seal is VALID\n", path);
    } else {
        printf("verify_chip: '%s' seal is INVALID\n", path);
    }

    return valid;
}

/* ============================================================================
 * PRINT_CHIP_INFO: Print chip information
 * ============================================================================ */

/*
 * Print all header fields and verification status.
 */
void print_chip_info(const char *path)
{
    if (!path) {
        fprintf(stderr, "print_chip_info: invalid path\n");
        return;
    }

    /* Open file */
    FILE *f = fopen(path, "rb");
    if (!f) {
        fprintf(stderr, "print_chip_info: failed to open '%s'\n", path);
        return;
    }

    /* Read header */
    xorc_header_t header;
    size_t read_bytes = fread(&header, 1, XORC_HEADER_SIZE, f);
    fclose(f);

    if (read_bytes != XORC_HEADER_SIZE) {
        fprintf(stderr, "print_chip_info: failed to read header\n");
        return;
    }

    /* Print header information */
    printf("=== XORC Chip: %s ===\n", path);

    /* Magic */
    printf("Magic:            %c%c%c%c\n",
           header.magic[0], header.magic[1], header.magic[2], header.magic[3]);

    /* Version */
    printf("Version:          %d.%d\n",
           (header.version >> 8) & 0xFF, header.version & 0xFF);

    /* Flags */
    printf("Flags:            0x%04X", header.flags);
    if (header.flags & XORC_FLAG_ASSOC) printf(" ASSOC");
    if (header.flags & XORC_FLAG_COMM) printf(" COMM");
    if (header.flags & XORC_FLAG_SEALED) printf(" SEALED");
    printf("\n");

    /* Dimension and size */
    printf("Dimension:        %d\n", header.dim);
    printf("Size (2^n):       %u\n", header.size);

    /* Family */
    printf("Family:           %s (%d)\n", family_name(header.family), header.family);

    /* Params */
    printf("Params:           %02X %02X\n", header.params[0], header.params[1]);

    /* Decode params based on family */
    switch (header.family) {
        case XORC_PAULI:
            printf("  Qubits:         %d\n", header.params[0]);
            break;
        case XORC_CLIFFORD:
            printf("  p:              %d\n", header.params[0]);
            printf("  q:              %d\n", header.params[1]);
            break;
        case XORC_CAYLEY:
            printf("  Level:          %d\n", header.params[0]);
            break;
        default:
            break;
    }

    /* Holonomy counts */
    printf("Assoc failures:   %u\n", header.assoc_fail);
    printf("Comm failures:    %u\n", header.comm_fail);

    /* Omega table info */
    printf("Omega offset:     %u\n", header.omega_off);
    printf("Omega size:       %u bytes\n", header.omega_sz);

    /* Seal (as hex) */
    printf("Seal:             ");
    for (int i = 0; i < 32; i++) {
        printf("%02x", header.seal[i]);
    }
    printf("\n");

    /* Verify seal */
    bool valid = verify_chip(path);
    printf("Seal status:      %s\n", valid ? "VALID" : "INVALID");

    /* Properties summary */
    printf("--- Properties ---\n");
    printf("Associative:      %s\n", (header.flags & XORC_FLAG_ASSOC) ? "yes" : "no");
    printf("Commutative:      %s\n", (header.flags & XORC_FLAG_COMM) ? "yes" : "no");
    printf("Sealed:           %s\n", (header.flags & XORC_FLAG_SEALED) ? "yes" : "no");

    printf("==========================\n");
}

/* ============================================================================
 * UTILITY FUNCTIONS FOR EXPORT PROCESSING
 * ============================================================================ */

/*
 * Export a registered kernel to a .xorc chip file.
 * This is a convenience function for the main compiler driver.
 *
 * Looks up the kernel and holonomy in the registry (defined in holo.c),
 * then calls codegen_chip with appropriate parameters.
 */

/* External declarations from holo.c registry */
extern xorc_omega_t* kernel_lookup(const char *name);
extern xorc_holo_t* kernel_holo(const char *name);

/*
 * Get family and params from a registered kernel.
 * This extracts the family information that was used to build the kernel.
 *
 * Note: This is a simplified version that infers family from properties.
 * A full implementation would store the family info in the registry.
 */
typedef struct {
    xorc_family_t family;
    codegen_params_t params;
} kernel_info_t;

/*
 * Export a named kernel to a chip file.
 * Uses the kernel registry to find omega and holo.
 *
 * The family and params must be provided since the registry
 * doesn't store this information.
 */
bool export_kernel(const char *name, const char *path,
                   xorc_family_t family, void *params)
{
    /* Look up kernel */
    xorc_omega_t *omega = kernel_lookup(name);
    if (!omega) {
        fprintf(stderr, "export: kernel '%s' not found\n", name);
        return false;
    }

    /* Look up holonomy */
    xorc_holo_t *holo = kernel_holo(name);
    if (!holo) {
        fprintf(stderr, "export: holonomy for '%s' not found\n", name);
        return false;
    }

    /* Generate chip file */
    return codegen_chip(path, omega, holo, family, params);
}

/* ============================================================================
 * CHIP_INFO: Get chip metadata without loading full omega
 * ============================================================================ */

/*
 * Get chip metadata without loading the full omega table.
 * Reads only the header and returns basic information.
 *
 * Returns true on success, false on error.
 */
bool chip_info(const char *path, xorc_dim_t *dim, xorc_family_t *family,
               uint32_t *assoc_fail, uint32_t *comm_fail)
{
    if (!path) {
        return false;
    }

    /* Open file */
    FILE *f = fopen(path, "rb");
    if (!f) {
        return false;
    }

    /* Read header */
    xorc_header_t header;
    size_t read_bytes = fread(&header, 1, XORC_HEADER_SIZE, f);
    fclose(f);

    if (read_bytes != XORC_HEADER_SIZE) {
        return false;
    }

    /* Verify magic */
    if (header.magic[0] != XORC_MAGIC_0 ||
        header.magic[1] != XORC_MAGIC_1 ||
        header.magic[2] != XORC_MAGIC_2 ||
        header.magic[3] != XORC_MAGIC_3) {
        return false;
    }

    /* Return metadata */
    if (dim) *dim = header.dim;
    if (family) *family = (xorc_family_t)header.family;
    if (assoc_fail) *assoc_fail = header.assoc_fail;
    if (comm_fail) *comm_fail = header.comm_fail;

    return true;
}

/* ============================================================================
 * CHIP COMPARISON UTILITY
 * ============================================================================ */

/*
 * Compare two chip files for equivalence.
 * Loads both chips and compares their omega tables.
 *
 * Returns:
 *  0 if chips are equivalent
 *  1 if chips differ
 * -1 on error
 */
int compare_chips(const char *path1, const char *path2)
{
    xorc_omega_t *omega1 = load_chip(path1);
    if (!omega1) {
        return -1;
    }

    xorc_omega_t *omega2 = load_chip(path2);
    if (!omega2) {
        omega_free(omega1);
        return -1;
    }

    /* Compare dimensions */
    if (omega1->dim != omega2->dim) {
        printf("compare: dimensions differ (%d vs %d)\n", omega1->dim, omega2->dim);
        omega_free(omega1);
        omega_free(omega2);
        return 1;
    }

    /* Compare omega tables */
    uint32_t size_sq = omega1->size * omega1->size;
    int result = 0;

    for (uint32_t i = 0; i < size_sq; i++) {
        if (omega1->data[i] != omega2->data[i]) {
            result = 1;
            break;
        }
    }

    omega_free(omega1);
    omega_free(omega2);

    if (result == 0) {
        printf("compare: chips '%s' and '%s' are equivalent\n", path1, path2);
    } else {
        printf("compare: chips '%s' and '%s' differ\n", path1, path2);
    }

    return result;
}

/* ============================================================================
 * CHIP DUMP UTILITY
 * ============================================================================ */

/*
 * Dump the omega table from a chip file to stdout.
 * Useful for debugging and inspection.
 */
void dump_chip_omega(const char *path)
{
    xorc_omega_t *omega = load_chip(path);
    if (!omega) {
        return;
    }

    printf("Omega table from '%s' (dim=%d, size=%u):\n", path, omega->dim, omega->size);

    if (omega->dim > 4) {
        printf("(table too large to display, showing first 16x16)\n");
        uint32_t display_size = 16;

        printf("     ");
        for (uint32_t v = 0; v < display_size && v < omega->size; v++) {
            printf("%3u ", v);
        }
        printf("\n");

        for (uint32_t u = 0; u < display_size && u < omega->size; u++) {
            printf("%3u: ", u);
            for (uint32_t v = 0; v < display_size && v < omega->size; v++) {
                xorc_sign_t s = omega_get(omega, u, v);
                printf(" %c  ", s > 0 ? '+' : '-');
            }
            printf("\n");
        }
    } else {
        printf("     ");
        for (uint32_t v = 0; v < omega->size; v++) {
            printf("%3u ", v);
        }
        printf("\n");

        for (uint32_t u = 0; u < omega->size; u++) {
            printf("%3u: ", u);
            for (uint32_t v = 0; v < omega->size; v++) {
                xorc_sign_t s = omega_get(omega, u, v);
                printf(" %c  ", s > 0 ? '+' : '-');
            }
            printf("\n");
        }
    }

    omega_free(omega);
}
